/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */


import { serializeIdentifier, type Id } from '../identifier.utils.js';
import { uint32ToBuf } from '../number.utils.js';

/**
 * Serializes partition parameters for create/delete operations.
 *
 * @param streamId - Stream identifier (ID or name)
 * @param topicId - Topic identifier (ID or name)
 * @param partitionCount - Number of partitions (1-1000)
 * @returns Buffer containing serialized partition request
 * @throws Error if partitionCount is not between 1 and 1000
 */
export const serializePartitionParams = (
  streamId: Id, topicId: Id, partitionCount = 1,
) => {

  if (partitionCount < 1 || partitionCount > 1000)
    throw new Error('Topic partition_count must be between 1 and 1000');

  const streamIdentifier = serializeIdentifier(streamId);
  const topicIdentifier = serializeIdentifier(topicId);
  const b = uint32ToBuf(partitionCount);

  return Buffer.concat([
    streamIdentifier,
    topicIdentifier,
    b,
  ])
};
