---
#
# Licensed to the Apache Software Foundation (ASF) under one
# or more contributor license agreements.  See the NOTICE file
# distributed with this work for additional information
# regarding copyright ownership.  The ASF licenses this file
# to you under the Apache License, Version 2.0 (the
# "License"); you may not use this file except in compliance
# with the License.  You may obtain a copy of the License at
#
#   http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing,
# software distributed under the License is distributed on an
# "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
# KIND, either express or implied.  See the License for the
# specific language governing permissions and limitations
# under the License.
#
title: Configuring Polaris
type: docs
weight: 550
---

## Overview

This page provides information on how to configure Apache Polaris (Incubating). Unless stated
otherwise, this information is valid both for Polaris Docker images (and Kubernetes deployments) as
well as for Polaris binary distributions.

{{< alert note >}}
For Production tips and best practices, refer to [Configuring Polaris for Production]({{% ref "configuring-polaris-for-production.md" %}}).
{{< /alert >}}

First off, Polaris server runs on Quarkus, and uses its configuration mechanisms. Read Quarkus
[configuration guide](https://quarkus.io/guides/config) to get familiar with the basics.

Quarkus aggregates configuration properties from multiple sources, applying them in a specific order
of precedence. When a property is defined in multiple sources, the value from the source with the
higher priority overrides those from lower-priority sources.

The sources are listed below, from highest to lowest priority:

1. System properties: properties set via the Java command line using `-Dproperty.name=value`.
2. Environment variables (see below for important details).
3. Settings in `$PWD/config/application.properties` file.
4. The `application.properties` files packaged in Polaris.
5. Default values: hardcoded defaults within the application.

When using environment variables, there are two naming conventions:

1. If possible, just use the property name as the environment variable name. This works fine in most
   cases, e.g. in Kubernetes deployments. For example, `polaris.realm-context.realms` can be
   included as is in a container YAML definition:
   ```yaml
   env:
   - name: "polaris.realm-context.realms"
     value: "realm1,realm2"
   ```

2. If running from a script or shell prompt, however, stricter naming rules apply: variable names
   can consist solely of uppercase letters, digits, and the `_` (underscore) sign. In such
   situations, the environment variable name must be derived from the property name, by using
   uppercase letters, and replacing all dots, dashes and quotes by underscores. For example,
   `polaris.realm-context.realms` becomes `POLARIS_REALM_CONTEXT_REALMS`. See
   [here](https://smallrye.io/smallrye-config/Main/config/environment-variables/) for more details.

{{< alert important >}}
While convenient, uppercase-only environment variables can be problematic for complex property
names. In these situations, it's preferable to use system properties or a configuration file.
{{< /alert >}}


As stated above, a configuration file can also be provided at runtime; it should be available
(mounted) at `$PWD/config/application.properties` for Polaris server to recognize it. In Polaris
official Docker images, this location is `/deployment/config/application.properties`.

For Kubernetes deployments, the configuration file is typically defined as a `ConfigMap`, then
mounted in the container at `/deployment/config/application.properties`. It can be mounted in
read-only mode, as Polaris only reads the configuration file once, at startup.

## Polaris Configuration Options Reference

| Configuration Property                                                                 | Default Value                       | Description                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                   |
|----------------------------------------------------------------------------------------|-------------------------------------|---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------|
| `polaris.persistence.type`                                                             | `relational-jdbc`                   | Define the persistence backend used by Polaris (`in-memory`, `relational-jdbc`). See [Configuring Apache Polaris for Production)[{{% ref "configuring-polaris-for-production.md" %}})                                                                                                                                                                                                                                                                                                                                                                                                                                            |
| `polaris.persistence.relational.jdbc.max-retries`                                      | `1`                                 | Total number of retries JDBC persistence will attempt on connection resets or serialization failures before giving up.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                        |
| `polaris.persistence.relational.jdbc.max_duaration_in_ms`                              | `5000 ms`                           | Max time interval (ms) since the start of a transaction when retries can be attempted.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                        |
| `polaris.persistence.relational.jdbc.initial_delay_in_ms`                              | `100 ms`                            | Initial delay before retrying. The delay is doubled after each retry.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                         |
| `polaris.realm-context.type`                                                           | `default`                           | Define the type of the Polaris realm to use.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                  |
| `polaris.realm-context.realms`                                                         | `POLARIS`                           | Define the list of realms to use.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                             |
| `polaris.realm-context.header-name`                                                    | `Polaris-Realm`                     | Define the header name defining the realm context.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                            |
| `polaris.features."ENFORCE_PRINCIPAL_CREDENTIAL_ROTATION_REQUIRED_CHECKING"`           | `false`                             | If set to true, require that principals must rotate their credentials before being used for anything else.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                    |
| `polaris.features."SKIP_CREDENTIAL_SUBSCOPING_INDIRECTION"`                            | `false`                             | If set to true, skip credential-subscoping indirection entirely whenever trying to obtain storage credentials for instantiating a FileIO. If 'true', no attempt is made to use StorageConfigs to generate table-specific storage credentials, but instead the default fallthrough of table-level credential properties or else provider-specific APPLICATION_DEFAULT credential-loading will be used for the FileIO. Typically this setting is used in single-tenant server deployments that don't rely on "credential-vending" and can use server-default environment variables or credential config files for all storage access, or in test/dev scenarios. |
| `polaris.features."ALLOW_SETTING_S3_ENDPOINTS"`                                        | `true`                              | If set to true (default), Polaris will permit S3 storage configurations to have custom endpoints. If set to false, Polaris will not accept catalog create and update requests that contain S3 endpoint properties.                                                                                                                                                                                                                                                                                                                                                                                                                                            |
| `polaris.features."ALLOW_TABLE_LOCATION_OVERLAP"`                                      | `false`                             | (DEPRECATION) If set to true, allow one table's location to reside within another table's location. This is only enforced within a given namespace.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                           |
| `polaris.features."ALLOW_NAMESPACE_LOCATION_OVERLAP"`                                  | `false`                             | If set to true, allow one namespace's location to reside within another namespace's location. This is only enforced within a parent catalog or namespace.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                     |
| `polaris.features."ALLOW_EXTERNAL_METADATA_FILE_LOCATION"`                             | `false`                             | If set to true, allows metadata files to be located outside the default metadata directory.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                   |
| `polaris.features."ALLOW_OVERLAPPING_CATALOG_URLS"`                                    | `false`                             | If set to true, allows catalog URLs to overlap.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                               |
| `polaris.features."ALLOW_UNSTRUCTURED_TABLE_LOCATION"`                                 | `false`                             | (DEPRECATION) If set to true, allows unstructured table locations.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                            |
| `polaris.features."ALLOW_EXTERNAL_TABLE_LOCATION"`                                     | `false`                             | (DEPRECATION) If set to true, allows tables to have external locations outside the default structure.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                         |
| `polaris.features."ALLOW_EXTERNAL_CATALOG_CREDENTIAL_VENDING"`                         | `true`                              | (DEPRECATION) If set to true, allow credential vending for external catalogs.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                 |
| `polaris.features."SUPPORTED_CATALOG_STORAGE_TYPES"`                                   | `S3`, `AZURE`, `GCS`                | (DEPRECATION) The list of supported storage types for a catalog.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                              |
| `polaris.features."CLEANUP_ON_NAMESPACE_DROP"`                                         | `false`                             | (DEPRECATION) If set to true, clean up data when a namespace is dropped.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                      |
| `polaris.features."CLEANUP_ON_CATALOG_DROP"`                                           | `false`                             | (DEPRECATION) If set to true, clean up data when a catalog is dropped.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                        |
| `polaris.features."DROP_WITH_PURGE_ENABLED"`                                           | `false`                             | (DEPRECATION) If set to true, allows tables to be dropped with the purge parameter set to true.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                               |
| `polaris.features."PURGE_VIEW_METADATA_ON_DROP"`                                       | `true`                              | If set to true, Polaris will attempt to delete view metadata files when a view is dropped.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                    |
| `polaris.features."STORAGE_CREDENTIAL_DURATION_SECONDS"`                               | `3600`                              | The duration of time that vended storage credentials are valid for. Support for longer (or shorter) durations is dependent on the storage provider. GCS current does not respect this value.                                                                                                                                                                                                                                                                                                                                                                                                                                                                  |
| `polaris.features."STORAGE_CREDENTIAL_CACHE_DURATION_SECONDS"`                         | `1800`                              | How long to store storage credentials in the local cache. This should be less than STORAGE_CREDENTIAL_DURATION_SECONDS.key().                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                 |
| `polaris.features."MAX_METADATA_REFRESH_RETRIES"`                                      | `2`                                 | How many times to retry refreshing metadata when the previous error was retryable.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                            |
| `polaris.features."LIST_PAGINATION_ENABLED"`                                           | `false`                             | If set to true, pagination for APIs like listTables is enabled.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                               |
| `polaris.features."ENABLE_GENERIC_TABLES"`                                             | `true`                              | If true, the generic-tables endpoints are enabled.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                            |
| `polaris.features."ENTITY_CACHE_WEIGHER_TARGET"`                                       | `104857600`                         | The maximum weight for the entity cache. This is a heuristic value without any particular unit of measurement. It roughly correlates with the total heap size of cached values. Fine-tuning requires experimentation in the specific deployment environment.                                                                                                                                                                                                                                                                                                                                                                                                  |
| `polaris.features."ENABLE_CATALOG_FEDERATION"`                                         | `false`                             | If true, allows creating and using ExternalCatalogs containing ConnectionConfigInfos to perform federation to remote catalogs.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                |
| `polaris.features."ENABLE_SUB_CATALOG_RBAC_FOR_FEDERATED_CATALOGS"`                    | `false`                             | When enabled, allows RBAC operations to create synthetic entities for entities in federated catalogs that don't exist in the local metastore.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                 |
| `polaris.features."ENABLE_POLICY_STORE"`                                               | `true`                              | If true, the policy-store endpoints are enabled.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                              |
| `polaris.features."SUPPORTED_CATALOG_CONNECTION_TYPES"`                                | `ICEBERG_REST`                      | The list of supported catalog connection types for federation.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                |
| `polaris.features."SUPPORTED_EXTERNAL_CATALOG_AUTHENTICATION_TYPES"`                   | `OAUTH`, `BEARER`, `SIGV4`          | The list of supported authentication types for catalog federation.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                            |
| `polaris.features."ICEBERG_COMMIT_MAX_RETRIES"`                                        | `4`                                 | The max number of times to try committing to an Iceberg table.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                |
| `polaris.features."ALLOW_SPECIFYING_FILE_IO_IMPL"`                                     | `false`                             | Config key for whether to allow setting the FILE_IO_IMPL using catalog properties. Must only be enabled in dev/test environments, should not be in production systems.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                        |
| `polaris.features."ALLOW_INSECURE_STORAGE_TYPES"`                                      | `false`                             | Allow usage of FileIO implementations that are considered insecure. Enabling this setting may expose the service to possibly severe security risks! This should only be set to 'true' for tests!                                                                                                                                                                                                                                                                                                                                                                                                                                                              |
| `polaris.features."ICEBERG_ROLLBACK_COMPACTION_ON_CONFLICTS"`                          | `false`                             | Rollback replace snapshots created by compaction which have polaris.internal.conflict-resolution.by-operation-type.replace property set to rollback in their snapshot summary.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                |
| `polaris.features."ADD_TRAILING_SLASH_TO_LOCATION"`                                    | `true`                              | When set, the base location for a table or namespace will have `/` added as a suffix if not present.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                          |
| `polaris.features."ALLOW_OPTIMIZED_SIBLING_CHECK"`                                     | `false`                             | When set to true, Polaris will permit enabling the feature OPTIMIZED_SIBLING_CHECK for catalogs, this is done to prevent accidental enabling the feature in cases such as schema migrations, without backfill and hence leading to potential data integrity issues. This will be removed in 2.0.0 when polaris ships with the necessary migrations to backfill the index.                                                                                                                                                                                                                                                                                     |
| `polaris.features."OPTIMIZED_SIBLING_CHECK"`                                           | `false`                             | When set, an index is used to perform the sibling check between tables, views, and namespaces. New locations will be checked against previous ones based on components, so the new location /foo/bar/ will check for a sibling at /, /foo/ and /foo/bar/%. In order for this check to be correct, locations should end with a slash. See ADD_TRAILING_SLASH_TO_LOCATION for a way to enforce this when new locations are added. Only supported by the JDBC metastore.                                                                                                                                                                                         |
| `polaris.features."DEFAULT_LOCATION_OBJECT_STORAGE_PREFIX_ENABLED"`                    | `false`                             | When enabled, Iceberg tables and views created without a location specified will have a prefix applied to the location within the catalog's base location, rather than a location directly inside the parent namespace. Note that this requires ALLOW_EXTERNAL_TABLE_LOCATION to be enabled, but with OPTIMIZED_SIBLING_CHECK enabled it is still possible to enforce the uniqueness of table locations within a catalog.                                                                                                                                                                                                                                     |
| `polaris.features."ENABLE_CREDENTIAL_RESET"`                                           | `true`                              | Flag to enable or disable the API to reset principal credentials. Defaults to enabled, but service providers may want to disable it.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                          |
| `polaris.features."ALLOW_SETTING_SUB_CATALOG_RBAC_FOR_FEDERATED_CATALOGS"`             | `true`                              | If set to true (default), Polaris will allow setting or changing catalog property polaris.config.enable-sub-catalog-rbac-for-federated-catalogs.If set to false, Polaris will disallow setting or changing the above catalog property.                                                                                                                                                                                                                                                                                                                                                                                                                        |
| `polaris.features."ALLOW_DROPPING_NON_EMPTY_PASSTHROUGH_FACADE_CATALOG"`               | `false`                             | If enabled, allow dropping a passthrough-facade catalog even if it contains namespaces or tables. passthrough-facade catalogs may contain leftover entities when syncing with source catalog.In the short term these entities will be ignored, in the long term there will be method/background job to clean them up.                                                                                                                                                                                                                                                                                                                                         |
| `polaris.features."ENABLE_FINE_GRAINED_UPDATE_TABLE_PRIVILEGES"`                       | `true`                              | When true, enables finer grained update table privileges which are passed to the authorizer for update table operations.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                      |
| `polaris.features."ALLOW_FEDERATED_CATALOGS_CREDENTIAL_VENDING"`                       | `true`                              | If set to true (default), allow credential vending for external catalogs. Note this requires ALLOW_EXTERNAL_CATALOG_CREDENTIAL_VENDING to be true first.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                      |
| `polaris.features.realm-overrides."my-realm"."SKIP_CREDENTIAL_SUBSCOPING_INDIRECTION"` | `true`                              | "Override" realm features, here the skip credential subscoping indirection flag.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                              |
| `polaris.authentication.type`                                                          | `internal`                          | The type of authentication to use. Three built-in types are supported: `internal`, `external`, and `mixed`.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                   |
| `polaris.authentication.authenticator.type`                                            | `default`                           | Define the Polaris authenticator type.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                        |
| `polaris.authentication.token-service.type`                                            | `default`                           | Define the Polaris token service type.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                        |
| `polaris.authentication.token-broker.type`                                             | `rsa-key-pair`                      | Define the Polaris token broker type. Also configure the location of the key files. For RSA: if the locations of the key files are not configured, an ephemeral key-pair will be created on each Polaris server instance startup, which breaks existing tokens after server restarts and is also incompatible with running multiple Polaris server instances.                                                                                                                                                                                                                                                                                                 |
| `polaris.authentication.token-broker.max-token-generation`                             | `PT1H`                              | Define the max token generation policy on the token broker.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                   |
| `polaris.authentication.token-broker.rsa-key-pair.private-key-file`                    |                                     | Define the location of the RSA-256 private key file, if present the `public-key` file must be specified, too.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                 |
| `polaris.authentication.token-broker.rsa-key-pair.public-key-file`                     |                                     | Define the location of the RSA-256 public key file, if present the `private-key` file must be specified, too.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                 |
| `polaris.authentication.token-broker.symmetric-key.secret`                             | `secret`                            | Define the secret of the symmetric key.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                       |
| `polaris.authentication.token-broker.symmetric-key.file`                               | `/tmp/symmetric.key`                | Define the location of the symmetric key file.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                |
| `polaris.oidc.principal-mapper.type`                                                   | `default`                           | Define the PrincipalMapper implementation. `default` extracts fields from claim paths.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                        |
| `polaris.oidc.principal-mapper.id-claim-path`                                          |                                     | Define the path to the claim containing the principal ID.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                     |
| `polaris.oidc.principal-mapper.name-claim-path`                                        |                                     | Define the path to the claim containing the principal name.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                   |
| `polaris.oidc.principal-roles-mapper.type`                                             | `default`                           | Define the PrincipalRolesMapper implementation. `default` applies regex transformations.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                      |
| `polaris.oidc.principal-roles-mapper.filter`                                           |                                     | Define the regex to filter roles from the OIDC token.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                         |
| `polaris.oidc.principal-roles-mapper.mappings[n].regex`                                |                                     | Define the regex to match a role for transformation.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                          |
| `polaris.oidc.principal-roles-mapper.mappings[n].replacement`                          |                                     | Define the replacement string for the matched role.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                           |
| `polaris.oidc.<tenant-id>.<property>=<value>`                                          |                                     | Define a per-tenant override for OIDC settings. Replace `<tenant-id>` with the tenant name, `<property>` with the property name and `<value>` with the property value.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                        |
| `polaris.storage.aws.access-key`                                                       | `accessKey`                         | Define the AWS S3 access key. If unset, the default credential provider chain will be used.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                   |
| `polaris.storage.aws.secret-key`                                                       | `secretKey`                         | Define the AWS S3 secret key. If unset, the default credential provider chain will be used.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                   |
| `polaris.storage.gcp.token`                                                            | `token`                             | Define the Google Cloud Storage token. If unset, the default credential provider chain will be used.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                          |
| `polaris.storage.gcp.lifespan`                                                         | `PT1H`                              | Define the Google Cloud Storage lifespan type. If unset, the default credential provider chain will be used.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                  |
| `polaris.log.request-id-header-name`                                                   | `X-Request-ID`                      | Define the header name to match request ID in the log.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                        |
| `polaris.log.mdc.aid`                                                                  | `polaris`                           | Define the log context (e.g. MDC) AID.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                        |
| `polaris.log.mdc.sid`                                                                  | `polaris-service`                   | Define the log context (e.g. MDC) SID.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                        |
| `polaris.rate-limiter.filter.type`                                                     | `no-op`                             | Define the Polaris rate limiter. Supported values are `no-op`, `token-bucket`.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                |
| `polaris.rate-limiter.token-bucket.type`                                               | `default`                           | Define the token bucket rate limiter.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                         |
| `polaris.rate-limiter.token-bucket.requests-per-second`                                | `9999`                              | Define the number of requests per second for the token bucket rate limiter.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                   |
| `polaris.rate-limiter.token-bucket.window`                                             | `PT10S`                             | Define the window type for the token bucket rate limiter.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                     |
| `polaris.metrics.tags.<tag-name>=<tag-value>`                                          | `application=Polaris`               | Define arbitrary metric tags to include in every request.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                      |
| `polaris.metrics.realm-id-tag.api-metrics-enabled`                                     | `false`                             | Whether to enable the `realm_id` metric tag in API metrics.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                   |
| `polaris.metrics.realm-id-tag.http-metrics-enabled`                                    | `false`                             | Whether to enable the `realm_id` metric tag in HTTP request metrics.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                          |
| `polaris.metrics.realm-id-tag.http-metrics-max-cardinality`                            | `100`                               | The maximum cardinality for the `realm_id` tag in HTTP request metrics.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                       |
| `polaris.tasks.max-concurrent-tasks`                                                   | `100`                               | Define the max number of concurrent tasks.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                    |
| `polaris.tasks.max-queued-tasks`                                                       | `1000`                              | Define the max number of tasks in queue.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                      |
| `polaris.config.rollback.compaction.on-conflicts.enabled`                              | `false`                             | When set to true Polaris will apply the deconfliction by rollbacking those REPLACE operations snapshots which have the property of `polaris.internal.rollback.compaction.on-conflict` in their snapshot summary set to `rollback`, to resolve conflicts at the server end.                                                                                                                                                                                                                                                                                                                                                                                    |
| `polaris.event-listener.type`                                                          | `no-op`                             | Define the Polaris event listener type. Supported values are `no-op`, `aws-cloudwatch`.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                       |
| `polaris.event-listener.aws-cloudwatch.log-group`                                      | `polaris-cloudwatch-default-group`  | Define the AWS CloudWatch log group name for the event listener.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                              |
| `polaris.event-listener.aws-cloudwatch.log-stream`                                     | `polaris-cloudwatch-default-stream` | Define the AWS CloudWatch log stream name for the event listener. Ensure that Polaris' IAM credentials have the following actions: "PutLogEvents", "DescribeLogStreams", and "DescribeLogGroups" on the specified log stream/group. If the specified log stream/group does not exist, then "CreateLogStream" and "CreateLogGroup" will also be required.                                                                                                                                                                                                                                                                                                      |
| `polaris.event-listener.aws-cloudwatch.region`                                         | `us-east-1`                         | Define the AWS region for the CloudWatch event listener.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                      |
| `polaris.event-listener.aws-cloudwatch.synchronous-mode`                               | `false`                             | Define whether log events are sent to CloudWatch synchronously. When set to true, events are sent synchronously which may impact performance but ensures immediate delivery. When false (default), events are sent asynchronously for better performance.                                                                                                                                                                                                                                                                                                                                                                                                     |
| `polaris.event-listener.persistence-in-memory-buffer.buffer-time`                      | `5000ms`                            | The duration to buffer events in memory before flushing for the persistence event listener.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                   |
| `polaris.event-listener.persistence-in-memory-buffer.max-buffer-size`                  | `5`                                 | The maximum number of events to hold in the in-memory buffer before triggering a flush.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                       |
| `polaris.credential-manager.type`                                                      | `default`                           | Define the credential manager config.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                         |

There are non Polaris configuration properties that can be useful:

| Configuration Property                               | Default Value                   | Description                                                                 |
|------------------------------------------------------|---------------------------------|-----------------------------------------------------------------------------|
| `quarkus.log.level`                                  | `INFO`                          | Define the root log level.                                                  |
| `quarkus.log.category."org.apache.polaris".level`    |                                 | Define the log level for a specific category.                               |
| `quarkus.default-locale`                             | System locale                   | Force the use of a specific locale, for instance `en_US`.                   |
| `quarkus.http.port`                                  | `8181`                          | Define the HTTP port number.                                                |
| `quarkus.http.auth.basic`                            | `false`                         | Enable the HTTP basic authentication.                                       |
| `quarkus.http.limits.max-body-size`                  | `10240K`                        | Define the HTTP max body size limit.                                        |
| `quarkus.http.cors.origins`                          |                                 | Define the HTTP CORS origins.                                               |
| `quarkus.http.cors.methods`                          | `PATCH, POST, DELETE, GET, PUT` | Define the HTTP CORS covered methods.                                       |
| `quarkus.http.cors.headers`                          | `*`                             | Define the HTTP CORS covered headers.                                       |
| `quarkus.http.cors.exposed-headers`                  | `*`                             | Define the HTTP CORS covered exposed headers.                               |
| `quarkus.http.cors.access-control-max-age`           | `PT10M`                         | Define the HTTP CORS access control max age.                                |
| `quarkus.http.cors.access-control-allow-credentials` | `true`                          | Define the HTTP CORS access control allow credentials flag.                 |
| `quarkus.management.enabled`                         | `true`                          | Enable the management server.                                               |
| `quarkus.management.port`                            | `8182`                          | Define the port number of the Polaris management server.                    |
| `quarkus.management.root-path`                       |                                 | Define the root path where `/metrics` and `/health` endpoints are based on. |
| `quarkus.otel.sdk.disabled`                          | `true`                          | Enable the OpenTelemetry layer.                                             |

{{< alert note >}}
This section is only relevant for Polaris Docker images and Kubernetes deployments.
{{< /alert >}}

There are many other actionable environment variables available in the official Polaris Docker
image; they come from the base image used by Polaris, [ubi9/openjdk-21-runtime]. They should be used
to fine-tune the Java runtime directly, e.g. to enable debugging or to set the heap size. These
variables are not specific to Polaris, but are inherited from the base image. If in doubt, leave
everything at its default!

[ubi9/openjdk-21-runtime]: https://catalog.redhat.com/software/containers/ubi9/openjdk-21-runtime/6501ce769a0d86945c422d5f

| Environment variable             | Description                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                            |
|----------------------------------|----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------|
| `JAVA_OPTS` or `JAVA_OPTIONS`    | **NOT RECOMMENDED**. JVM options passed to the `java` command (example: "-verbose:class"). Setting this variable will override all options set by any of the other variables in this table. To pass extra settings, use `JAVA_OPTS_APPEND` instead.                                                                                                                                                                                                                                                                                    |
| `JAVA_OPTS_APPEND`               | User specified Java options to be appended to generated options in `JAVA_OPTS` (example: "-Dsome.property=foo").                                                                                                                                                                                                                                                                                                                                                                                                                       |
| `JAVA_TOOL_OPTIONS`              | This variable is defined and honored by all OpenJDK distros, see [here](https://bugs.openjdk.org/browse/JDK-4971166). Options defined here take precedence over all else; using this variable is generally not necessary, but can be useful e.g. to enforce JVM startup parameters, to set up remote debug, or to define JVM agents.                                                                                                                                                                                                   |
| `JAVA_MAX_MEM_RATIO`             | Is used to calculate a default maximal heap memory based on a containers restriction. If used in a container without any memory constraints for the container then this option has no effect. If there is a memory constraint then `-XX:MaxRAMPercentage` is set to a ratio of the container available memory as set here. The default is `80` which means 80% of the available memory is used as an upper boundary. You can skip this mechanism by setting this value to `0` in which case no `-XX:MaxRAMPercentage` option is added. |
| `JAVA_DEBUG`                     | If set remote debugging will be switched on. Disabled by default (example: true").                                                                                                                                                                                                                                                                                                                                                                                                                                                     |
| `JAVA_DEBUG_PORT`                | Port used for remote debugging. Defaults to "5005" (tip: use "*:5005" to enable debugging on all network interfaces).                                                                                                                                                                                                                                                                                                                                                                                                                  |
| `GC_MIN_HEAP_FREE_RATIO`         | Minimum percentage of heap free after GC to avoid expansion. Default is 10.                                                                                                                                                                                                                                                                                                                                                                                                                                                            |
| `GC_MAX_HEAP_FREE_RATIO`         | Maximum percentage of heap free after GC to avoid shrinking. Default is 20.                                                                                                                                                                                                                                                                                                                                                                                                                                                            |
| `GC_TIME_RATIO`                  | Specifies the ratio of the time spent outside the garbage collection. Default is 4.                                                                                                                                                                                                                                                                                                                                                                                                                                                    |
| `GC_ADAPTIVE_SIZE_POLICY_WEIGHT` | The weighting given to the current GC time versus previous GC times. Default is 90.                                                                                                                                                                                                                                                                                                                                                                                                                                                    |
| `GC_METASPACE_SIZE`              | The initial metaspace size. There is no default (example: "20").                                                                                                                                                                                                                                                                                                                                                                                                                                                                       |
| `GC_MAX_METASPACE_SIZE`          | The maximum metaspace size. There is no default (example: "100").                                                                                                                                                                                                                                                                                                                                                                                                                                                                      |
| `GC_CONTAINER_OPTIONS`           | Specify Java GC to use. The value of this variable should contain the necessary JRE command-line options to specify the required GC, which will override the default of `-XX:+UseParallelGC` (example: `-XX:+UseG1GC`).                                                                                                                                                                                                                                                                                                                |
Here are some examples:

| Example                                    | `docker run` option                                                                                                 |
|--------------------------------------------|---------------------------------------------------------------------------------------------------------------------|
| Using another GC                           | `-e GC_CONTAINER_OPTIONS="-XX:+UseShenandoahGC"` lets Polaris use Shenandoah GC instead of the default parallel GC. |
| Set the Java heap size to a _fixed_ amount | `-e JAVA_OPTS_APPEND="-Xms8g -Xmx8g"` lets Polaris use a Java heap of 8g.                                           |
| Set the maximum heap percentage            | `-e JAVA_MAX_MEM_RATIO="70"` lets Polaris use 70% percent of the available memory.                                  |


## Troubleshooting Configuration Issues

If you encounter issues with the configuration, you can ask Polaris to print out the configuration it
is using. To do this, set the log level for the `io.smallrye.config` category to `DEBUG`, and also
set the console appender level to `DEBUG`:

```properties
quarkus.log.console.level=DEBUG
quarkus.log.category."io.smallrye.config".level=DEBUG
```

{{< alert important >}}
This will print out all configuration values, including sensitive ones like
passwords. Don't do this in production, and don't share this output with anyone you don't trust!
{{< /alert >}}
