/*
 *  $Id: tip.c 28909 2025-11-24 18:04:22Z yeti-dn $
 *  Copyright (C) 2003-2025 David Necas (Yeti), Petr Klapetek.
 *  E-mail: yeti@gwyddion.net, klapetek@gwyddion.net.
 *
 *  This program is free software; you can redistribute it and/or modify it under the terms of the GNU General Public
 *  License as published by the Free Software Foundation; either version 2 of the License, or (at your option) any
 *  later version.
 *
 *  This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied
 *  warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
 *  details.
 *
 *  You should have received a copy of the GNU General Public License along with this program; if not, write to the
 *  Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

#include "config.h"
#include <string.h>
#include <glib/gi18n-lib.h>

#include "libgwyddion/macros.h"
#include "libgwyddion/stats.h"
#include "libgwyddion/tip.h"

#include "libgwyddion/omp.h"
#include "libgwyddion/morph-lib.h"
#include "libgwyddion/internal.h"

#define MORPH_LIB_N 10000.0

static gint**
field_to_intfield(GwyField *field,
                  gboolean maxzero,
                  gdouble min,
                  gdouble step)
{
    gint **ret;
    gint col, row;
    gdouble max;

    max = maxzero ? gwy_field_get_max(field) : 0.0;

    gint xres = field->xres, yres = field->yres;
    const gdouble *data = field->priv->data;
    ret = _gwy_morph_lib_iallocmatrix(field->yres, xres);
    for (row = 0; row < yres; row++) {
        for (col = 0; col < xres; col++)
            ret[row][col] = (gint)(((data[col + xres*row] - max) - min)/step);
    }

    return ret;
}

static void
intfield_to_field(gint **field,
                  GwyField *ret,
                  gdouble min,
                  gdouble step)
{
    gint col, row;

    gint xres = ret->xres, yres = ret->yres;
    gdouble *d = ret->priv->data;
    for (row = 0; row < yres; row++) {
        for (col = 0; col < xres; col++)
            d[xres*row + col] = field[row][col]*step + min;
    }

    gwy_field_invalidate(ret);
}

static gint**
field_to_largeintfield(GwyField *field,
                       GwyField *tipfield,
                       gdouble min,
                       gdouble step)
{
    gint **ret;
    gint col, row;
    gint xres, yres, xnew, ynew;
    gint txr2, tyr2;
    gint minimum;
    gdouble *data;

    minimum = (gint)((gwy_field_get_min(field) - min)/step);
    xres = field->xres;
    yres = field->yres;
    xnew = xres + tipfield->xres;
    ynew = yres + tipfield->yres;
    txr2 = tipfield->xres/2;
    tyr2 = tipfield->yres/2;

    data = field->priv->data;
    ret = _gwy_morph_lib_iallocmatrix(ynew, xnew);
    for (row = 0; row < ynew; row++) {
        for (col = 0; col < xnew; col++) {
            if (col >= txr2
                && col < xres + txr2
                && row >= tyr2
                && row < yres + tyr2)
                ret[row][col] = (gint)(((data[col - txr2 + xres*(row - tyr2)])
                                        - min) /step);
            else
                ret[row][col] = minimum;
        }
    }

    return ret;
}

static GwyField*
largeintfield_to_field(gint **field,
                       GwyField *ret,
                       GwyField *tipfield,
                       gdouble min,
                       gdouble step)
{
    gint col, row;
    gint xnew, ynew;
    gint txr2, tyr2;

    xnew = ret->xres + tipfield->xres;
    ynew = ret->yres + tipfield->yres;
    txr2 = tipfield->xres/2;
    tyr2 = tipfield->yres/2;

    gdouble *data = ret->priv->data;
    for (row = 0; row < ynew; row++) {
        for (col = 0; col < xnew; col++) {
            if (col >= txr2 && col < (ret->xres + txr2) && row >= tyr2 && row < (ret->yres + tyr2))
                data[col - txr2 + ret->xres*(row - tyr2)] = field[row][col]*step + min;
        }
    }

    gwy_field_invalidate(ret);
    return ret;
}

static GwyField*
get_right_tip_field(GwyField *tip,
                    GwyField *surface)
{
    gint xres, yres;

    xres = GWY_ROUND(tip->xreal/surface->xreal*surface->xres);
    xres = MAX(xres, 1);
    yres = GWY_ROUND(tip->yreal/surface->yreal*surface->yres);
    yres = MAX(yres, 1);

    return gwy_field_new_resampled(tip, xres, yres, GWY_INTERPOLATION_BSPLINE);
}

static inline gdouble
tip_convolve_interior(const gdouble *src, gint xres,
                      const gdouble *tip, gint txres, gint tyres)
{
    gdouble hmax = -G_MAXDOUBLE;
    gint i, j;

    for (i = 0; i < tyres; i++) {
        const gdouble *srcrow = src + i*xres;
        for (j = txres; j; j--) {
            gdouble h = *(srcrow++) + *(tip++);
            if (h > hmax)
                hmax = h;
        }
    }
    return hmax;
}

static inline gdouble
tip_convolve_border(const gdouble *src, gint xres, gint yres,
                    const gdouble *tip, gint txres, gint tyres,
                    gint j, gint i)
{
    gint ioff = tyres/2, joff = txres/2;
    gdouble hmax = -G_MAXDOUBLE;
    gint ii, jj;

    for (ii = 0; ii < tyres; ii++) {
        gint isrc = CLAMP(i + ii - ioff, 0, yres-1);
        for (jj = 0; jj < txres; jj++) {
            gint jsrc = CLAMP(j + jj - joff, 0, xres-1);
            gdouble h = src[isrc*xres + jsrc] + *(tip++);
            if (h > hmax)
                hmax = h;
        }
    }
    return hmax;
}

/**
 * gwy_tip_dilation:
 * @tip: Tip data.
 * @surface: Surface data.
 * @result: Data field where to store dilated surface to.
 * @set_fraction: (scope call) (nullable): Function that sets fraction to output (or %NULL).
 * @set_message: (scope call) (nullable): Function that sets message to output (or %NULL).
 *
 * Performs the tip convolution algorithm published by Villarrubia, which is
 * equivalent to morphological dilation operation.
 *
 * If the operation is aborted the size and contents of @result field is
 * undefined.
 *
 * Returns: (transfer none): Dilated surface data, i.e. @result, on success.  May return %NULL
 *          if aborted.
 **/
GwyField*
gwy_tip_dilation(GwyField *tip,
                 GwyField *surface,
                 GwyField *result,
                 GwySetFractionFunc set_fraction,
                 GwySetMessageFunc set_message)
{
    gint txres, tyres, xres, yres, ioff, joff;
    const gdouble *sdata, *tdata;
    gdouble *data;
    GwyField *mytip;
    gboolean cancelled = FALSE, *pcancelled = &cancelled;

    g_return_val_if_fail(GWY_IS_FIELD(tip), NULL);
    g_return_val_if_fail(GWY_IS_FIELD(surface), NULL);
    g_return_val_if_fail(GWY_IS_FIELD(result), NULL);

    if ((set_message && !set_message(_("Dilation...")))
        || (set_fraction && !set_fraction(0.0)))
        return NULL;

    xres = surface->xres;
    yres = surface->yres;
    gwy_field_resize(result, xres, yres);
    gwy_field_invalidate(result);

    /* Preserve the surface height as original implementation does. */
    mytip = gwy_field_copy(tip);
    gwy_field_add(mytip, -gwy_field_get_max(mytip));

    txres = tip->xres;
    tyres = tip->yres;
    sdata = surface->priv->data;
    tdata = mytip->priv->data;
    data = result->priv->data;
    ioff = tyres/2;
    joff = txres/2;

#ifdef _OPENMP
#pragma omp parallel if (gwy_threads_are_enabled()) default(none) \
            shared(sdata,data,tdata,xres,yres,txres,tyres,ioff,joff,set_fraction,pcancelled)
#endif
    {
        gint ifrom = gwy_omp_chunk_start(yres), ito = gwy_omp_chunk_end(yres);
        gint i, j;

        for (i = ifrom; i < ito; i++) {
            gboolean row_inside = (i >= ioff && i + tyres-ioff <= yres);

            for (j = 0; j < xres; j++) {
                gboolean col_inside = (j >= joff && j + txres-joff <= xres);
                gint k = i*xres + j;

                if (row_inside && col_inside) {
                    const gdouble *src = sdata + (i - ioff)*xres + (j - joff);
                    data[k] = tip_convolve_interior(src, xres, tdata, txres, tyres);
                }
                else
                    data[k] = tip_convolve_border(sdata, xres, yres, tdata, txres, tyres, j, i);
            }

            if (gwy_omp_set_fraction_check_cancel(set_fraction, i, ifrom, ito, pcancelled))
                break;
        }
    }

    g_object_unref(mytip);
    return cancelled ? NULL : result;
}

static inline gdouble
tip_erode_interior(const gdouble *src, gint xres,
                   const gdouble *tip, gint txres, gint tyres)
{
    gdouble hmin = G_MAXDOUBLE;
    gint i, j;

    for (i = 0; i < tyres; i++) {
        const gdouble *srcrow = src + i*xres;
        for (j = txres; j; j--) {
            gdouble h = *(srcrow++) - *(tip++);
            if (h < hmin)
                hmin = h;
        }
    }
    return hmin;
}

static inline gdouble
tip_erode_border(const gdouble *src, gint xres, gint yres,
                 const gdouble *tip, gint txres, gint tyres,
                 gint j, gint i)
{
    gint ioff = tyres/2, joff = txres/2;
    gdouble hmin = G_MAXDOUBLE;
    gint ii, jj;

    for (ii = 0; ii < tyres; ii++) {
        gint isrc = CLAMP(i + ii - ioff, 0, yres-1);
        for (jj = 0; jj < txres; jj++) {
            gint jsrc = CLAMP(j + jj - joff, 0, xres-1);
            gdouble h = src[isrc*xres + jsrc] - *(tip++);
            if (h < hmin)
                hmin = h;
        }
    }
    return hmin;
}

/**
 * gwy_tip_erosion:
 * @tip: Tip data.
 * @surface: Surface to be eroded.
 * @result: Data field where to store dilated surface to.
 * @set_fraction: (scope call) (nullable): Function that sets fraction to output (or %NULL).
 * @set_message: (scope call) (nullable): Function that sets message to output (or %NULL).
 *
 * Performs surface reconstruction (erosion) algorithm published by
 * Villarrubia, which is equivalent to morphological erosion operation.
 *
 * If the operation is aborted the size and contents of @result field is
 * undefined.
 *
 * Returns: (transfer none): Reconstructed (eroded) surface, i.e. @result, on success.  May
 *          return %NULL if aborted.
 **/
GwyField*
gwy_tip_erosion(GwyField *tip,
                GwyField *surface,
                GwyField *result,
                GwySetFractionFunc set_fraction,
                GwySetMessageFunc set_message)
{
    gint txres, tyres, xres, yres, ioff, joff;
    const gdouble *sdata, *tdata;
    GwyField *mytip;
    gdouble *data;
    gboolean cancelled = FALSE, *pcancelled = &cancelled;

    g_return_val_if_fail(GWY_IS_FIELD(tip), NULL);
    g_return_val_if_fail(GWY_IS_FIELD(surface), NULL);
    g_return_val_if_fail(GWY_IS_FIELD(result), NULL);

    if ((set_message && !set_message(_("Erosion...")))
        || (set_fraction && !set_fraction(0.0)))
        return NULL;

    xres = surface->xres;
    yres = surface->yres;
    gwy_field_resize(result, xres, yres);

    /* Preserve the surface height as original implementation does. */
    mytip = gwy_field_copy(tip);
    gwy_field_flip(mytip, TRUE, TRUE);
    gwy_field_add(mytip, -gwy_field_get_max(mytip));

    txres = tip->xres;
    tyres = tip->yres;
    sdata = surface->priv->data;
    tdata = mytip->priv->data;
    data = result->priv->data;
    ioff = tyres/2;
    joff = txres/2;

#ifdef _OPENMP
#pragma omp parallel if (gwy_threads_are_enabled()) default(none) \
            shared(sdata,data,tdata,xres,yres,txres,tyres,ioff,joff,set_fraction,pcancelled)
#endif
    {
        gint ifrom = gwy_omp_chunk_start(yres), ito = gwy_omp_chunk_end(yres);
        gint i, j;

        for (i = ifrom; i < ito; i++) {
            gboolean row_inside = (i >= ioff && i + tyres-ioff <= yres);

            for (j = 0; j < xres; j++) {
                gboolean col_inside = (j >= joff && j + txres-joff <= xres);
                gint k = i*xres + j;

                if (row_inside && col_inside) {
                    const gdouble *src = sdata + (i - ioff)*xres + (j - joff);
                    data[k] = tip_erode_interior(src, xres, tdata, txres, tyres);
                }
                else
                    data[k] = tip_erode_border(sdata, xres, yres, tdata, txres, tyres, j, i);
            }

            if (gwy_omp_set_fraction_check_cancel(set_fraction, i, ifrom, ito, pcancelled))
                break;
        }
    }

    g_object_unref(mytip);
    return cancelled ? NULL : result;
}

/**
 * gwy_tip_cmap:
 * @tip: Tip data.
 * @surface: Surface data.
 * @result: Data field to store ceratainty map data to.
 * @set_fraction: (scope call) (nullable): Function that sets fraction to output (or %NULL).
 * @set_message: (scope call) (nullable): Function that sets message to output (of %NULL).
 *
 * Performs certainty map algorithm published by Villarrubia. This function
 * converts all fields into form requested by "morph_lib.c" library, that is
 * almost identical with original Villarubia's library. Result certainty map
 * can be used as a mask of points where tip did not directly touch the
 * surface.
 *
 * Returns: (transfer none): Certainty map, i.e. @result, on success.  May return %NULL if
 *          aborted.
 **/
#if 0
static void
certainty_map(GwyField *image,
              GwyField *tip,
              GwyField *reconstructed,
              GwyField *cmap,
              gint xc, gint yc,
              gdouble threshold)
{
    gint i, j, rxc, ryc;              /* center coordinates of reflected tip */
    gint xres, yres, txres, tyres;
    const gdouble *data, *tdata, *rdata;
    gdouble *cdata;

    xres = image->xres;
    yres = image->yres;
    data = image->data;
    rdata = reconstructed->data;
    txres = tip->xres;
    tyres = tip->yres;
    tdata = tip->data;
    g_return_if_fail(reconstructed->xres == xres);
    g_return_if_fail(reconstructed->yres == yres);
    g_return_if_fail(cmap->xres == xres);
    g_return_if_fail(cmap->yres == yres);

    rxc = txres-1 - xc;
    ryc = tyres-1 - yc;

    gwy_field_clear(cmap);
    cdata = cmap->data;

    /* Loop over all pixels in the interior of the image. We skip pixels near
     * the edge. Since it is possible there are unseen touches over the edge,
     * we must conservatively leave these cmap entries at 0. */
#ifdef _OPENMP
#pragma omp parallel for if (gwy_threads_are_enabled()) default(none) \
            private(i,j) \
            shared(data,tdata,rdata,cdata,xres,yres,txres,tyres,rxc,ryc,threshold)
#endif
    for (i = ryc; i <= yres + ryc - tyres; i++) {
        for (j = rxc; j <= xres + rxc - txres; j++) {
            gint tjmin = MAX(0, rxc - j);
            gint tjmax = MIN(txres-1, xres-1 + rxc - j);
            gint timin = MAX(0, ryc - i);
            gint timax = MIN(tyres-1, yres-1 + ryc - i);
            gint x, y, ti, tj, count = 0;
            gdouble z = data[i*xres + j];

            for (ti = timin; ti <= timax && count < 2; ti++) {
                const gdouble *trow = tdata + (tyres-1 - ti)*txres;
                const gdouble *rrow = rdata + (ti + i - ryc)*xres + (j - rxc);

                for (tj = tjmin; tj <= tjmax; tj++) {
                    /* Use a difference threshold for `exact' contact instead
                     * of relying on discretisation to integers.  This does
                     * not exactly reproduce the discretisation behaviour... */
                    if (fabs(z - trow[txres-1-tj] - rrow[tj]) <= threshold) {
                        /* Remember coordinates. */
                        x = tj + j - rxc;
                        y = ti + i - ryc;
                        /* Increment count. */
                        count++;
                        if (count > 1)
                            break;
                    }
                }
            }
            /* One contact = good recon. */
            /* This is OK with parallelisation because if we write from
             * multiple threads to the same location, we write the same
             * value. */
            if (count == 1)
                cdata[y*xres + x] = 1.0;
        }
    }
}

GwyField*
gwy_tip_cmap(GwyField *tip,
             GwyField *surface,
             GwyField *cmap,
             G_GNUC_UNUSED GwySetFractionFunc set_fraction,
             G_GNUC_UNUSED GwySetMessageFunc set_message)
{
    gdouble min, max, step;
    gint xres, yres, extx, exty, extx2, exty2;
    GwyField *buffertip, *extsurface, *rsurface;

    g_return_val_if_fail(GWY_IS_FIELD(tip), NULL);
    g_return_val_if_fail(GWY_IS_FIELD(surface), NULL);
    g_return_val_if_fail(GWY_IS_FIELD(cmap), NULL);
    xres = surface->xres;
    yres = surface->yres;

    /*if tip and surface have different spacings, make new, resampled tip*/
    buffertip = get_right_tip_field(tip, surface);
    /*invert tip (as necessary by dilation algorithm)*/
    gwy_field_get_min_max(buffertip, &min, &max);
    gwy_field_flip(buffertip, TRUE, TRUE);
    /* FIXME: No idea why we subtract min+max.  It compensates some strange
     * offset in the certainty map... */
    gwy_field_add(buffertip, -(min + max));

    gwy_field_get_min_max(surface, &min, &max);
    /* This is the discretisation step used in blind tip estimation, so
     * assume it is reasonable as a threshold here. */
    step = (max - min)/MORPH_LIB_N;

    extx = buffertip->xres/2;
    exty = buffertip->yres/2;
    extx2 = buffertip->xres - extx;
    exty2 = buffertip->yres - exty;
    extsurface = gwy_field_extend(surface, extx, extx2, exty, exty2,
                                  GWY_EXTERIOR_FIXED_VALUE, min, FALSE);
    rsurface = gwy_field_new_alike(surface, FALSE);
    gwy_tip_erosion(buffertip, extsurface, rsurface, NULL, NULL);
    g_object_unref(extsurface);
    gwy_field_resize(cmap, extsurface->xres, extsurface->yres);
    certainty_map(extsurface, buffertip, rsurface, cmap, extx, exty, 50.0*step);
    g_object_unref(rsurface);
    g_object_unref(buffertip);

    gwy_field_resize(cmap, extx, exty, extx + xres, exty + yres);
    gwy_field_copy_units(surface, cmap);
    gwy_unit_clear(gwy_field_get_unit_z(cmap));

    return cmap;
}
#else
GwyField*
gwy_tip_cmap(GwyField *tip,
             GwyField *surface,
             GwyField *result,
             GwySetFractionFunc set_fraction,
             GwySetMessageFunc set_message)
{
    gint newx, newy, bxres, byres;
    gdouble tipmin, surfacemin, step;
    GwyField *buffertip;

    /*if tip and surface have different spacings, make new, resampled tip*/
    buffertip = get_right_tip_field(tip, surface);
    bxres = buffertip->xres;
    byres = buffertip->yres;
    /*invert tip (as necessary by dilation algorithm)*/
    gwy_field_flip(buffertip, TRUE, TRUE);

    newx = surface->xres + bxres;
    newy = surface->yres + byres;

    /*convert fields to integer arrays*/
    tipmin = gwy_field_get_min(buffertip);
    surfacemin = gwy_field_get_min(surface);
    step = (gwy_field_get_max(surface) - surfacemin)/MORPH_LIB_N;

    gint **ftip = field_to_intfield(buffertip, TRUE, tipmin, step);
    gint **fsurface = field_to_largeintfield(surface, buffertip, surfacemin, step);

    /*perform erosion as it is necessary parameter of certainty map algorithm*/
    gint **rsurface = _gwy_morph_lib_ierosion((const gint* const*)fsurface, newx, newy,
                                              (const gint* const*)ftip, bxres, byres,
                                              bxres/2, byres/2,
                                              set_message, set_fraction);

    /*find certanty map*/
    gint **fresult = NULL;
    if (rsurface) {
        fresult = _gwy_morph_lib_icmap((const gint* const*)fsurface, newx, newy,
                                       (const gint* const*)ftip, bxres, byres,
                                       (const gint* const*)rsurface,
                                       bxres/2, byres/2,
                                       set_message, set_fraction);
    }

    /*convert result back*/
    if (fresult) {
        gwy_field_resize(result, surface->xres, surface->yres);
        largeintfield_to_field(fresult, result, buffertip, 0.0, 1.0);
    }
    else
        result = NULL;

    g_object_unref(buffertip);
    _gwy_morph_lib_free_intfield(ftip);
    _gwy_morph_lib_free_intfield(fsurface);
    _gwy_morph_lib_free_intfield(rsurface);
    _gwy_morph_lib_free_intfield(fresult);

    return result;
}
#endif

/**
 * gwy_tip_estimate_partial:
 * @tip: Tip data to be refined (allocated).
 * @surface: Surface data.
 * @threshold: Threshold for noise supression.
 * @use_edges: Whether use also edges of image.
 * @count: (out): Where to store the number of places that produced refinements to.
 * @set_fraction: (scope call) (nullable): Function that sets fraction to output (or %NULL).
 * @set_message: (scope call) (nullable): Function that sets message to output (or %NULL).
 *
 * Performs partial blind estimation algorithm published by Villarrubia. This
 * function converts all fields into form requested by "morph_lib.c" library,
 * that is almost identical with original Villarubia's library. Note that the
 * threshold value must be chosen sufficently high value to supress small
 * fluctulations due to noise (that would lead to very sharp tip) but
 * sufficiently low value to put algorithm at work. A value similar to 1/10000
 * of surface range can be good. Otherwise we recommend to start with zero
 * threshold and increase it slowly to observe changes and choose right value.
 *
 * Returns: (transfer none): Estimated tip.  May return %NULL if aborted.
 **/
GwyField*
gwy_tip_estimate_partial(GwyField *tip,
                         GwyField *surface,
                         gdouble threshold,
                         gboolean use_edges,
                         gint *count,
                         GwySetFractionFunc set_fraction,
                         GwySetMessageFunc set_message)
{
    gdouble tipmin, surfacemin, step;
    gint cnt;

    tipmin = gwy_field_get_min(tip);
    surfacemin = gwy_field_get_min(surface);
    step = (gwy_field_get_max(surface)-surfacemin)/MORPH_LIB_N;

    gint **ftip = field_to_intfield(tip, TRUE,  tipmin, step);
    gint **fsurface = field_to_intfield(surface, FALSE, surfacemin, step);

    if (set_message && !set_message(N_("Starting partial estimation"))) {
        _gwy_morph_lib_free_intfield(ftip);
        _gwy_morph_lib_free_intfield(fsurface);
        return NULL;
    }

    cnt = _gwy_morph_lib_itip_estimate0((const gint* const*)fsurface,
                                        surface->xres, surface->yres,
                                        tip->xres, tip->yres,
                                        tip->xres/2, tip->yres/2,
                                        ftip, threshold/step, use_edges,
                                        set_message, set_fraction);
    if (cnt == -1 || (set_fraction && !set_fraction(0.0))) {
        _gwy_morph_lib_free_intfield(ftip);
        _gwy_morph_lib_free_intfield(fsurface);
        return NULL;
    }

    intfield_to_field(ftip, tip, tipmin, step);
    gwy_field_add(tip, -gwy_field_get_min(tip));

    _gwy_morph_lib_free_intfield(ftip);
    _gwy_morph_lib_free_intfield(fsurface);
    if (count)
        *count = cnt;

    return tip;
}


/**
 * gwy_tip_estimate_full:
 * @tip: Tip data to be refined (allocated).
 * @surface: Surface data.
 * @threshold: Threshold for noise supression.
 * @use_edges: Whether use also edges of image.
 * @count: (out): Where to store the number of places that produced refinements to.
 * @set_fraction: (scope call) (nullable): Function that sets fraction to output (or %NULL).
 * @set_message: (scope call) (nullable): Function that sets message to output (or %NULL).
 *
 * Performs full blind estimation algorithm published by Villarrubia. This
 * function converts all fields into form requested by "morph_lib.c" library,
 * that is almost identical with original Villarubia's library. Note that the
 * threshold value must be chosen sufficently high value to supress small
 * fluctulations due to noise (that would lead to very sharp tip) but
 * sufficiently low value to put algorithm at work. A value similar to 1/10000
 * of surface range can be good. Otherwise we recommend to start with zero
 * threshold and increase it slowly to observe changes and choose right value.
 *
 * Returns: (transfer none): Estimated tip.  May return %NULL if aborted.
 **/
GwyField*
gwy_tip_estimate_full(GwyField *tip,
                      GwyField *surface,
                      gdouble threshold,
                      gboolean use_edges,
                      gint *count,
                      GwySetFractionFunc set_fraction,
                      GwySetMessageFunc set_message)
{
    gdouble tipmin, surfacemin, step;
    gint cnt;

    tipmin = gwy_field_get_min(tip);
    surfacemin = gwy_field_get_min(surface);
    step = (gwy_field_get_max(surface) - surfacemin)/MORPH_LIB_N;

    gint **ftip = field_to_intfield(tip, TRUE, tipmin, step);
    gint **fsurface = field_to_intfield(surface, FALSE,  surfacemin, step);

    if (set_message && !set_message(N_("Starting full estimation..."))) {
        _gwy_morph_lib_free_intfield(ftip);
        _gwy_morph_lib_free_intfield(fsurface);
        return NULL;
    }

    cnt = _gwy_morph_lib_itip_estimate((const gint *const*)fsurface,
                                       surface->xres, surface->yres,
                                       tip->xres, tip->yres,
                                       tip->xres/2, tip->yres/2,
                                       ftip, threshold/step, use_edges,
                                       set_message, set_fraction);
    if (cnt == -1 || (set_fraction && !set_fraction(0.0))) {
        _gwy_morph_lib_free_intfield(ftip);
        _gwy_morph_lib_free_intfield(fsurface);
        return NULL;
    }

    intfield_to_field(ftip, tip, tipmin, step);
    gwy_field_add(tip, -gwy_field_get_min(tip));

    _gwy_morph_lib_free_intfield(ftip);
    _gwy_morph_lib_free_intfield(fsurface);
    if (count)
        *count = cnt;

    return tip;
}

/**
 * SECTION: tip
 * @title: Tip
 * @short_description: SPM tip morphological operations
 **/

/* vim: set cin columns=120 tw=118 et ts=4 sw=4 cino=>1s,e0,n0,f0,{0,}0,^0,\:1s,=0,g1s,h0,t0,+1s,c3,(0,u0 : */
